function [Gamma, GammaD, logProb, logProbD, PostGamD, MargGam, SWITCH]=...
bvsgs_spgi(gamprec,X,Y, delta,k, w, v1,pm,nDum)

%bvsgs_spgi: Bayesian Variable Selection - Gibbs Sampler - Main Program
%	     Selection prior (that is, v_0j = 0, j=1...p)
%	     Models with interactions and groups of variables 
%***********************************************************************
% INPUTS:
% gamprec,      starting binary vector.
%               If gamprec=[] the program asks for r to select a starting
%               vector with first r elements equal to 1 
% X,            independent variables, matrix (n by p)
% Y,            response variables, matrix (n by q)
% delta, k,     hyperparameters Inverse Wishart
% w,   		Bernoulli prior
%               In the case of STRONG HEREDITY, 
%               w=[wm p11 eta wmD p11D], where
%               wm = hyperparameter Bernoulli prior for main effects
%               p11 = p(ga_jk=1|ga_j=ga_k=1)(j may be equal to k)
%               eta= p(ga_jk=1|ga_j = 0 or ga_k=0)
%               wmD = prior probability of a group of variable to be
%               active.
%               p11D = p(ga_jk=1|ga_j=ga_k=1)(when j or k is a group of
%               variable).
%
%               In the case of WEAK HEREDITY,
%               w = [wm, p11, p10, p00, wmD, p11D p10D], with
%               wm = hyperparameter Bernoulli prior for independent main
%               effects,
%               p11 = p(ga_ik =1|ga_i=1,ga_k=1) for i,k independent effects;
%               p10 = p(ga_ik=1|ga_i=0,ga_k=1)=p(ga_ik=1|ga_i=1,ga_k=0)
%               for  i,k independent effects;
%               p00 = p(ga_ik=1|ga_i=0, ga_k=0)for independent effect or when
%               i or k is a group of variables,
%               wmD = prior probability of a group of variable to be
%               active,
%               p11D = p(ga_ik =1|ga_i=1,ga_k=1) when i or k is a group of
%               variables,
%               p10D = p(ga_ik=1|ga_i=0,ga_k=1)=p(ga_ik=1|ga_i=1,ga_k=0)
%               when i or k is a group of variables.
% 
% v1,		hyperparameter - normal selection prior 
%		(column vector (p by 1) of standard deviations)
% pm,		Number of independent main effects (i.e. not the variables
%		that are in a group)
% nDum,		vector whose components give the actual number of 
%		variables in each group.
%
% OUTPUTS:
% Gamma,        all visited vectors - matrix in sparse form
%		(Gamma(1,:) contains the starting vector)
% GammaD,	distinct visited vectors, ordered according to
%		their (normalized) relative posterior probability
%		(matrix in sparse form)
% logProb,      log-relative posterior probabilities 
%		of all visited vectors
% logProbD	log-relative posterior probabilities
%               of distinct visited vectors
% PostGamD	normalized ordered relative probabilities 
%		of distinct visited vectors 
% MargGam	marginal probabilities of components 
% SWITCH,	number of component switches (out of p) from iteration 
%		to iteration.
%		
%USAGE:
% [Gamma, GammD, logProb, logProbD, PostGamD, MargGam, SWITCH]= ...
%	bvsgs_sp(gamprec, X,Y, delta,k, w, v1)
%
%NOTES:
% Data must be centered.
% The independent effects (main effects and interactions) must be
% provided first.
% After the independent effects, the order is as follows: if, for
% example, nDum=[3 2], the order is :
% D_11, D_12,D_13,D_21,D_22 (main effects of grouped variables),
% X_1D_11,X_1D_12,X_1D_13 (first independent effect multiplied by the
% variables of the first group),...,
% X_pmD_11,X_pmD_12,X_pmD_13 ( last independent effect multiplied by the
% variables of the first group), X_1D_21,X_1D_22(first independent effect
% multiplied by the variables of the second group),...,X_pmD_21,X_pmD_22 
% (last independent effect multiplied by the variables of the second 
% group). 
% The programs asks for the Gibbs parameters (initial number of variables
% included, number of iterations)
% QR matrices updated every m iterations (m provided by the user)
% Gamma and GammaD are in sparse form.
% No permutation of the data is allowed.
% Functions called: gofg_sp, gibbs_sp, repliche, probord
%
%REFERENCE: 
% Brown, P.J., Vannucci, M. and Fearn, T.
% Multivariate Bayesian variable selection and prediction
% Journal of the Royal Statistical Society B, 60(3), 1998, pp. 627-641. 
% 
% Delouille,V., Bayesian variable selection with related predictors,
% MSc thesis, University of Kent at Canterbury, UK, 1998.
%
%Copyright (c) 1997 Marina Vannucci.
%Modified by Veronique Delouille 1998
%**********************************************************************
[nOss p]=size(X);
[q]=size(Y,2); 
lw=length(w);
if (lw~=5 & lw~=7)
disp('  w must be of length 5 for strong heredity principle')
disp('  or of length 7 for the weak heredity principle')
w=input('  Please provide w, w=?');
end;   
                % p independent variables, q responses, nOss observations
disp(' ')
disp('------- GIBBS SAMPLER --------')
disp(' ')
inde = order12(pm);
pg=shorten(inde,nDum,pm);
disp(' Note: number of components in gamprec must be equal to ')
pg
indici=1:p;
indicig=1:pg;
if isempty(gamprec)
	disp(' ')	
	disp('-- Starting binary vector with first r elements equal to 1');
	r=input('   please provide r <= pg, r=?');	
	gamprec=[ones(1,r) zeros(1,pg-r)];
else
	gamprec=gamprec(indicig);
end

disp(' ')
disp('-- To avoid build up of rounding errors,') 
disp('   QR matrices are recomputed every m iterations')
nPar=input('   please provide  m, m=?'); 
disp(' ')
ngibbs=input('-- Number ot total iterations (multiple than m)=?');

disp(' ')
disp('------- RUNNING GIBBS SAMPLER ...')
disp(' ')

v1=v1(indici);       	% standard deviations (mixture normal prior)
Gamma=sparse(zeros(ngibbs+1, pg));
logProb=zeros(1,ngibbs+1);
Gamma(1,indicig)=gamprec;
SWITCH=zeros(1,ngibbs);
conta=1;
Ytilde=cat(1,Y, zeros(p,q));      % augmented response matrix

while conta<ngibbs+1

%-------------------- updating QR decomposition
	[probprec, Qqr, Rqr] = ...
	gofg_sp(gamprec,1,X,Ytilde,[],[],[],k,delta,w,v1,inde,nDum,pm);
	logProb(conta)= probprec;

%-------------------- Gibbs (nPar iterations)
        [ProbSt, GammaSt, storeS]= gibbs_sp( ...
	gamprec,probprec,X,Ytilde,Qqr,Rqr,k,delta,w,v1,nPar,inde,nDum,pm);
	Gamma(conta+1:conta+nPar,indicig)=GammaSt;
	logProb(conta+1:conta+nPar)=ProbSt;
	SWITCH(conta:conta+nPar-1)=storeS;
	conta=conta+nPar;
	gamprec=GammaSt(nPar,:);
	disp(' ')
        disp('Number of iterations:')
        disp(conta-1)
        disp('...')
end

disp(' ')
disp('------- Searching for replicates ...')
disp(' ')
[GammaD, logProbD]=repliche(Gamma, logProb);

%------------------- Normalized posterior and marginal probs.
%		     Ordering of distinct visited vectors according
%                    to probability

[GammaD, logProbD, PostGamD,MargGam]=probord(logProbD,GammaD,inde,nDum,pm);

disp(' ')
disp('--- End.') 
