function [Gamma, GammaD, logProb, logProbD, PostGamD, MargGam, SWITCH]= ...
bvsgs_spi(gamprec,X,Y, delta,k, w, v1)

%bvsgs_spi: Bayesian Variable Selection - Gibbs Sampler - Main Program
%	    Selection prior (that is, v_0j = 0, j=1...p)
%	    Models with interactions : Bernoulli prior for gamma with 
%           strong or weak heredity assumptions. 
%***********************************************************************
%INPUTS:
% gamprec,      starting binary vector.
%               If gamprec=[] the program asks for r to select a starting
%               vector with first r elements equal to 1 
% X,            independent variables, matrix (n by p)
% Y,            response variables, matrix (n by q)
% delta, k,     hyperparameters Inverse Wishart
% w,   		Bernoulli prior, 
%		for STRONG HEREDITY principle, w is 
%		decomposed as [wm p eta] where:
%               wm = hyperparam Bernoulli prior for main effects
%               p = hyperparam Bernoulli prior for
%                   p(g_ik=1|g_i=g_k=1)(i may be equal to k)
%               eta = hyperparam Bernoulli prior for
%                   p(g_ik=1|g_i = 0 and/or g_k=0), eta small
%		for WEAK HEREDITY principle w is 
%		decomposed as  w=[wm, p11, p10, p00], where
%        	wm = hyperparameter Bernoulli prior for main effects
%         	p11 = p(g_ik =1|g_i=1,g_k=1);
%         	p10 = p(g_ik=1|g_i=0,g_k=1)=p(g_ik=1|g_i=1,g_k=0);
%       	p00 = p(g_ik=1|g_i=0, g_k=0), p00 small;
%
% 
% v1,		hyperparameter - normal selection prior of coefficients
%		(column vector (p by 1) of standard deviations)
%		(e.g. v1=c*sqrt(diag(pinv(X'*X))), with c a 'well chosen'
%		constant, for example c=1, 0.1 or 10)
%OUTPUTS:
% Gamma,        all visited vectors - matrix in sparse form
%		(Gamma(1,:) contains the starting vector)
% GammaD,	distinct visited vectors, ordered according to
%		their (normalized) relative posterior probability
%		(matrix in sparse form)
% logProb,      log-relative posterior probabilities 
%		of all visited vectors
% logProbD	log-relative posterior probabilities
%               of distinct visited vectors
% PostGamD	normalized ordered relative probabilities 
%		of distinct visited vectors 
% MargGam	marginal probabilities of components 
% SWITCH,	number of component switches (out of p) from iteration 
%		to iteration.
%		
%USAGE:
% [Gamma, GammD, logProb, logProbD, PostGamD, MargGam, SWITCH]= ...
%	bvsgs_sp(gamprec, X,Y, delta,k, w, v1,pm)
%
%NOTES:
% Data must be centered. 
% In the matrix X, the columns variables must be ordered as
% X_1,...,X_pm,X_1^2, X_1X_2, X_1X_3,...,X_1X_pm,X_2^2,X_2X_3,...,X_pm^2.
% (The program processing.m generates the second-order terms given the
% independent effects X_1,...,X_pm).
% QR matrices updated every m iterations (m provided by the user).
% The programs asks for the total number of iterations and for the number
% of independent main effects (pm).
% Gamma and GammaD are in sparse form.
% Functions called: gofg_sp, gibbs_sp, repliche, probord
%
%REFERENCE: 
% Brown, P.J., Vannucci, M. and Fearn, T.
% Multivariate Bayesian variable selection and prediction
% Journal of the Royal Statistical Society B, 60(3), 1998, pp. 627-641. 
%
% Delouille,V., Bayesian variable selection with related predictors,
% MSc thesis, University of Kent at Canterbury, UK, 1998.
%
%Copyright (c) 1997 Marina Vannucci. 
%Modified by Veronique Delouille 1998.
%**********************************************************************
[nOss p]=size(X);
[q]=size(Y,2); 
lw=length(w);
if (lw~=3 & lw~=4)
disp('  w must be a vector of length 3 for strong heredity principle')
disp('  or of length 4 for the weak heredity principle')
w=input('  Please provide w, w=?');
end;

                % p independent variables, q responses,nOss observations
disp(' ')
disp('------- GIBBS SAMPLER --------')
disp(' ')
indici=1:p;
if isempty(gamprec)
	disp(' ')
	disp('-- Starting binary vector with first r elements equal to 1');
	r=input('   please provide r, r=?');	
	gamprec=[ones(1,r) zeros(1,p-r)];
else
	gamprec=gamprec(indici);
end

disp(' ')
disp('-- To avoid build up of rounding errors,') 
disp('   QR matrices are recomputed every m iterations')
nPar=input('   please provide  m, m=?'); 
disp(' ')
ngibbs=input('-- Number ot total iterations (multiple than m)=?');
pm=input('-- Number of independent main effects =?');

disp(' ')
disp('------- RUNNING GIBBS SAMPLER ...')
disp(' ')

v1=v1(indici);       	% standard deviations (mixture normal prior)
Gamma=sparse(zeros(ngibbs+1, p));
logProb=zeros(1,ngibbs+1);
Gamma(1,indici)=gamprec;
SWITCH=zeros(1,ngibbs);
conta=1;
Ytilde=cat(1,Y, zeros(p,q));      % augmented response matrix

while conta<ngibbs+1

%-------------------- updating QR decomposition
	[probprec, Qqr, Rqr] = ...
		gofg_sp(gamprec,1,X,Ytilde,[],[],[],k,delta,w,v1,pm);
	logProb(conta)= probprec;
%-------------------- Gibbs (nPar iterations)
        [ProbSt, GammaSt, storeS]= gibbs_sp( ...
	gamprec,probprec,X,Ytilde,Qqr,Rqr,k,delta,w,v1,nPar,pm);
	Gamma(conta+1:conta+nPar,indici)=GammaSt;
	logProb(conta+1:conta+nPar)=ProbSt;
	SWITCH(conta:conta+nPar-1)=storeS;
	conta=conta+nPar;
	gamprec=GammaSt(nPar,:);
	disp(' ')
        disp('Number of iterations:')
        disp(conta-1)
        disp('...')
end

disp(' ')
disp('------- Searching for replicates ...')
disp(' ')
[GammaD, logProbD]=repliche(Gamma, logProb);

%------------------- Normalized posterior and marginal probs.
%		     Ordering of distinct visited vectors according
%                    to probability

[GammaD, logProbD, PostGamD, MargGam]=probord(logProbD, GammaD);

disp(' ')
disp('--- End.') 
