\name{HaseltineRawlings}
\Rdversion{1.1}
\alias{HaseltineRawlings}
\alias{GillespieOptimDirect}
\alias{RungeKuttaDormandPrince45}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Simulation of a biochemical system parameterized as a Petri Net
}
\description{These functions simulate Biochemical Petri Nets.
  \code{GillespieOptimDirect} performs a pure stochastic simulation,
  \code{RungeKuttaDormandPrince45} a pure deterministic run, while
  \code{HaseltineRawlings} a hybrid of the above. Multiple runs
  can be performed at once.
}
\usage{
GillespieOptimDirect(pre, post, h, M, T, delta = 1, runs = 1, place = NA, transition = NA)
RungeKuttaDormandPrince45(pre, post, h, M, T, delta = 1, place = NA, transition = NA, ect = 1e-09)
HaseltineRawlings(pre, post, h, slow, M, T, delta = 1, runs = 1, place = NA, transition = NA, ect = 1e-09)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{pre}{pre matrix, with as many rows as transitions, and
    columns as places. It has the stoichiometrics of the left sides of
    the reactions}
  \item{post}{post matrix, with as many rows as transitions, and
    columns as places. It has the stoichiometrics of the right sides of
    the reactions
}
  \item{h}{list of propensity constants or function returning the
    propensity (with as many elements as transitions)
}
  \item{slow}{vector of zeros for slow transitions and ones for fast transitions 
}
  \item{M}{Initial marking of the system
}
  \item{T}{Length in seconds that the process will run. It can be an R
    or C function
}
  \item{delta}{Interval length in which the samples will be saved (in seconds)
}
  \item{runs}{How many runs will be performed
}
  \item{place}{vector with names of the places
}
  \item{transition}{vector with names of the transitions
}
  \item{ect}{Precision for the fast reactions
}
}
\details{
%%  ~~ If necessary, more details than the description above ~~
}
\value{
The functions return a list with the following elements:  
  \item{place }{vector with the names of the places if supplied. If not,
  the function creates names as follows: P1, P2, ...}
  \item{transition }{vector with the names of the transitions if supplied. If not,
  the function creates names as follows: T1, T2, ...}
  \item{dt }{vector containing the discretized times at which the state
    is saved (according to delta)}
  \item{run }{list with as many elements as runs. We will describe
    the first element, run[[1]], as the rest have exactly the
    same structure. It is also a list, with the following elements:}
  \item{run[[1]]$M }{data frame where each row has the marking (state) of the system
    at the corresponding discretized times, so run[[1]]$M has as many rows as elements has
    dt. The column names are the same as place.}
  \item{run[[1]]$transitions }{vector with as many elements as transitions,
    with the total of time each slow reaction fired.}
  \item{run[[1]]$tot.transitions }{numeric with the summation of run[[1]]$transitions.}
}
\references{
%% ~put references to the literature/web site here ~
A complete manual with a more complex example is available at
www.stat.rice.edu/~mathbio/bioPN. It explains how to define functions
in R or C for ad-hoc propensities, and to create time protocols (change
on behaviour of the run at certain time points).
The example presented here is based on
Paszek, P. (2007) 
Modeling stochasticity in gene regulation: 
characterization in the terms of the underlying distribution function, 
Bull Math Biol, 69, 1567-1601.
}
\author{Roberto Bertolusso, Marek Kimmel
}
\note{
%%  ~~further notes~~
}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
%% ~~objects to See Also as \code{\link{help}}, ~~~
}
\examples{
    #########################################################
###### First approach, minimum requirements to run a model ######
    #########################################################
library(bioPN)
pre <- matrix(c(1,0,0,0, 0,1,0,0, 0,1,0,0,
                0,0,1,0, 0,0,1,0, 0,0,0,1), ncol=4,byrow=T)
post <- matrix(c(0,1,0,0, 1,0,0,0, 0,1,1,0,
                 0,0,0,0, 0,0,1,1, 0,0,0,0), ncol=4,byrow=T)
h <- list(c=3, b=2, H=10, 1, K=6, r=0.25)
M <- c(1,0,0,0)
T <- 200
delta <- 1

## Completely deterministic run
Sim <- RungeKuttaDormandPrince45(pre, post, h, M, T, delta)

runs <- 10
## Completely stochastic run
set.seed(19761111)  ## Set a seed
Sim <- GillespieOptimDirect(pre, post, h, M, T, delta, runs)

## Hybrid run
slow <- c(1,1,0,0,0,0)
set.seed(19761111)  ## Set a seed
Sim <- HaseltineRawlings(pre, post, h, slow, M, T, delta, runs)

    #####################################################
###### Second approach, more suitable for large models ######
    #####################################################
library(bioPN)

T <- 200
delta <- 1
ect <- 1e-9

place <- c( "I", "A", "mRNA", "protein")

transition <- c("gene_activation", "gene_inactivation",
                "transcription", "mRNA_degradation",
                "translation", "protein_degradation")

places <- length(place)
for (n in 1:places) {
  assign(place[n], n)
}
transitions <- length(transition)
for (n in 1:transitions) {
  assign(transition[n], n)
}
pre <- post <- array(0, dim <- c(transitions, places))
h <- list()

M <- rep(0, places)
M[I] <- 1

## Constants for Prokariotes
H <- 10
K <- 6
r <- 0.25
c <- 3
b <- 2
   
## Gene Activation: I -> A   (c)
i <- gene_activation
h[[i]] <- c
pre[i, I] <- 1
post[i, A] <- 1

## Gene Inactivation: A -> I   (d)
i <- gene_inactivation
h[[i]] <- b
pre[i, A] <- 1
post[i, I] <- 1
  
## Transcription: A -> A + mRNA   (H)
i <- transcription
h[[i]] <- H
pre[i, A] <- 1
post[i, A] <- 1; post[i, mRNA] <- 1
  
## mRNA Degradation: mRNA -> 0   (1)
i <- mRNA_degradation
h[[i]] <- 1
pre[i, mRNA] <- 1
  
## Translation: mRNA -> mRNA + protein   (K)
i <- translation
h[[i]] <- K
pre[i, mRNA] <- 1
post[i, mRNA] <- 1; post[i, protein] <- 1
  
## Protein Degradation: mRNA -> 0   (r)
i <- protein_degradation
h[[i]] <- r
pre[i, protein] <- 1

##############################
## Completely Deterministic ##
##############################
Sim <- RungeKuttaDormandPrince45(pre, post, h, M,
                                 T, delta, place, transition, ect)

## Note, it could also be done as follows
## slow <- rep(0, transitions)
## Sim <- HaseltineRawlings(pre, post, h, slow, M,
##                          T, delta, runs = 1, ect, place, transition)

mRNA.run <- Sim$run[[1]]$M$mRNA
protein.run <- Sim$run[[1]]$M$protein

## Theoretical results (red lines in following plots)
Mean.mRNA <- c/(c+b)*H
Mean.protein <- Mean.mRNA * K/r

par(mfrow=c(1,2))
par(mar=c(2, 4, 2, 1) + 0.1)
plot(Sim$dt, mRNA.run,type="l", ylab="Mean",main="mRNA")
legend(x="bottom", paste("Deterministic run"))
abline(h=Mean.mRNA,col="red", lwd=1)
plot(Sim$dt, protein.run,type="l", ylab="Mean",main="Protein")
legend(x="bottom", paste("Deterministic run"))
abline(h=Mean.protein,col="red", lwd=1)


runs <- 100	## Increase to 10000 to see better fit
###########################
## Completely Stochastic ##
###########################
set.seed(19761111)
Sim <- GillespieOptimDirect(pre, post, h, M, T, delta, runs, place, transition)

## Note, it could also be done as follows
## slow <- rep(1, transitions)
## Sim <- HaseltineRawlings(pre, post, h, slow, M,
##                          T, delta, runs, ect, place, transition)
    
mRNA.run <- sapply(Sim$run, function(run) {run$M$mRNA})
protein.run <- sapply(Sim$run, function(run) {run$M$protein})

## Histograms of protein at different time points.
par(mfrow=c(2,2))
par(mar=c(2, 4, 2.5, 1) + 0.1)
hist(protein.run[Sim$dt == 1,], main="Protein Distribution at t=1sec")
hist(protein.run[Sim$dt == 2,], main="Protein Distribution at t=2sec")
hist(protein.run[Sim$dt == 10,], main="Protein Distribution at t=10sec")
hist(protein.run[Sim$dt == 200,], main="Protein Distribution at t=200sec")

## Theoretical results (red lines in following plots)
Mean.mRNA <- c/(c+b)*H
Var.mRNA <- b/(c*(1+c+b))*Mean.mRNA^2 + Mean.mRNA
Mean.protein <- Mean.mRNA * K/r
Var.protein <- r*b*(1+c+b+r)/(c*(1+r)*(1+c+b)*(r+c+b))*Mean.protein^2 +
  r/(1+r)*Mean.protein^2/Mean.mRNA + Mean.protein

if (runs > 1 ) {
  par(mfrow=c(2,2))
} else {
  par(mfrow=c(1,2))
}
par(mar=c(2, 4, 2, 1) + 0.1)
plot(Sim$dt, apply(mRNA.run,1,function(tpt) {mean(tpt)}),type="l", ylab="Mean",main="mRNA")
legend(x="bottom", paste("Gene, mRNA and Protein Stochastic\nRuns :", runs))
abline(h=Mean.mRNA,col="red", lwd=1)
plot(Sim$dt, apply(protein.run,1,function(tpt) {mean(tpt)}),type="l", ylab="Mean",main="Protein")
legend(x="bottom", paste("Gene, mRNA and Protein Stochastic\nRuns :", runs))
abline(h=Mean.protein,col="red", lwd=1)
if (runs > 1 ) {
  par(mar=c(2, 4, 0, 1) + 0.1)
  plot(Sim$dt, apply(mRNA.run,1,function(tpt) {var(tpt)}),type="l", ylab="Var")
  abline(h=Var.mRNA,col="red", lwd=1)
  plot(Sim$dt, apply(protein.run,1,function(tpt) {var(tpt)}),type="l", ylab="Var")
  abline(h=Var.protein,col="red", lwd=1)
}

######################################################################
## Hybrid: mRNA and protein fast, gene activation/inactivation slow ##
######################################################################
slow <- rep(0, transitions)
slow[gene_activation] <- 1
slow[gene_inactivation] <- 1

set.seed(19761111)
Sim <- HaseltineRawlings(pre, post, h, slow, M,
                         T, delta, runs, place, transition, ect)
    
mRNA.run <- sapply(Sim$run, function(run) {run$M$mRNA})
protein.run <- sapply(Sim$run, function(run) {run$M$protein})
    
Mean.mRNA <- c/(c+b)*H
Var.mRNA <- b/(c*(1+c+b))*Mean.mRNA^2
Mean.protein <- Mean.mRNA * K/r
Var.protein <- r*b*(1+c+b+r)/(c*(1+r)*(1+c+b)*(r+c+b))*Mean.protein^2

if (runs > 1 ) {
  par(mfrow=c(2,2))
} else {
  par(mfrow=c(1,2))
}
par(mar=c(2, 4, 2, 1) + 0.1)
plot(Sim$dt, apply(mRNA.run,1,function(tpt) {mean(tpt)}),type="l", ylab="Mean",main="mRNA")
legend(x="bottom", paste("Only Gene Stochastic\nRuns :", runs))
abline(h=Mean.mRNA,col="red", lwd=1)
plot(Sim$dt, apply(protein.run,1,function(tpt) {mean(tpt)}),type="l", ylab="Mean",main="Protein")
legend(x="bottom", paste("Only Gene Stochastic\nRuns :", runs))
abline(h=Mean.protein,col="red", lwd=1)
if (runs > 1 ) {
  par(mar=c(2, 4, 0, 1) + 0.1)
  plot(Sim$dt, apply(mRNA.run,1,function(tpt) {var(tpt)}),type="l", ylab="Var")
  abline(h=Var.mRNA,col="red", lwd=1)
  plot(Sim$dt, apply(protein.run,1,function(tpt) {var(tpt)}),type="l", ylab="Var")
  abline(h=Var.protein,col="red", lwd=1)
}
    
######################################################################
## Hybrid: protein fast, mRNA and gene activation/inactivation slow ##
######################################################################
slow <- rep(1, transitions)
slow[translation] <- 0
slow[protein_degradation] <- 0

set.seed(19761111)
Sim <- HaseltineRawlings(pre, post, h, slow, M,
                         T, delta, runs, place, transition, ect)
    
mRNA.run <- sapply(Sim$run, function(run) {run$M[[mRNA]]})
protein.run <- sapply(Sim$run, function(run) {run$M[[protein]]})

Mean.mRNA <- c/(c+b)*H
Var.mRNA <- b/(c*(1+c+b))*Mean.mRNA^2 + Mean.mRNA
Mean.protein <- Mean.mRNA * K/r
Var.protein <- r*b*(1+c+b+r)/(c*(1+r)*(1+c+b)*(r+c+b))*Mean.protein^2 +
  r/(1+r)*Mean.protein^2/Mean.mRNA
    
if (runs > 1 ) {
  par(mfrow=c(2,2))
} else {
  par(mfrow=c(1,2))
}
par(mar=c(2, 4, 2, 1) + 0.1)
plot(Sim$dt, apply(mRNA.run,1,function(tpt) {mean(tpt)}),type="l", ylab="Mean",main="mRNA")
legend(x="bottom", paste("Gene and mRNA Stochastic\nRuns :", runs))
abline(h=Mean.mRNA,col="red", lwd=1)
plot(Sim$dt, apply(protein.run,1,function(tpt) {mean(tpt)}),type="l", ylab="Mean",main="Protein")
legend(x="bottom", paste("Gene and mRNA Stochastic\nRuns :", runs))
abline(h=Mean.protein,col="red", lwd=1)
par(mar=c(2, 4, 0, 1) + 0.1)
if (runs > 1 ) {
  plot(Sim$dt, apply(mRNA.run,1,function(tpt) {var(tpt)}),type="l", ylab="Var")
  abline(h=Var.mRNA,col="red", lwd=1)
  plot(Sim$dt, apply(protein.run,1,function(tpt) {var(tpt)}),type="l", ylab="Var")
  abline(h=Var.protein,col="red", lwd=1)
}
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ ~kwd1 }
\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line
