
/* Copyright (C) 2009  Roberto Bertolusso, Marek Kimmel

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, a copy is available at
   http://www.r-project.org/Licenses/
*/

#include <R.h>
#include <Rinternals.h>
#include <Rmath.h>
#include <R_ext/Rdynload.h>
#include <R_ext/Visibility.h>

#include "quicksort.h"
#include "helperfunctions.h"

#define PRODUCTS_MAXIMUM_AMOUNT 100000000
#define HAZARD_VALUE_TO_IGNORE 1e-10

#define RB_TIME
#define RB_SUBTIME

#ifdef RB_TIME
#include <time.h>
#endif


SEXP rb_gillespie(SEXP pre, SEXP post, SEXP h, SEXP M, SEXP steps)
{
  int k, iTmp, iProtected = 0;

#ifdef RB_TIME
  clock_t c0, c1;
  c0 = clock();
#endif

  // Get dimensions of pre
  int *piTmp = INTEGER(getAttrib(pre, R_DimSymbol));
  int iTransitions = piTmp[0], iPlaces = piTmp[1];

  int *piPre = INTEGER(pre), *piPost = INTEGER(post);

  double *pdH = REAL(h);

  int iTransition, iPlace, iTransitionPtr, iPlacePtr,
    iTransition2, iTransitionPtr2;

  // Setup Matrix S
  SEXP sexpTmp;
  PROTECT(sexpTmp = allocMatrix(INTSXP, iTransitions, iPlaces));
  ++iProtected;
  int *piS = INTEGER(sexpTmp);

  // Position of non zero cells in pre per place
  PROTECT(sexpTmp = allocMatrix(INTSXP, iTransitions, iPlaces));
  ++iProtected;
  int *piPreNZxCol = INTEGER(sexpTmp);

  // Totals of non zero cells in pre per place
  PROTECT(sexpTmp = allocVector(INTSXP, iPlaces));
  ++iProtected;
  int *piPreNZxColTot = INTEGER(sexpTmp);

  for (iPlace = 0; iPlace < iPlaces; iPlace++) {
    int iPreNZxCol_row = 0;
    for (iTransition = 0; iTransition < iTransitions; iTransition++) {
      if (piPre[iTransition + iTransitions * iPlace]) {
	piPreNZxCol[iPreNZxCol_row + iTransitions * iPlace] = iTransition;
	iPreNZxCol_row++;
      }
      piS[iTransition + iTransitions * iPlace] = 
	piPost[iTransition + iTransitions * iPlace] - piPre[iTransition + iTransitions * iPlace];
    }
    piPreNZxColTot[iPlace] = iPreNZxCol_row;
  }

  // Position of non zero cells in pre per transition
  PROTECT(sexpTmp = allocMatrix(INTSXP, iTransitions, iPlaces));
  ++iProtected;
  int *piPreNZxRow = INTEGER(sexpTmp);

  // Totals of non zero cells in pre per transition
  PROTECT(sexpTmp = allocVector(INTSXP, iTransitions));
  ++iProtected;
  int *piPreNZxRowTot = INTEGER(sexpTmp);

  // Position of non zero cells in S per transition
  PROTECT(sexpTmp = allocMatrix(INTSXP, iTransitions, iPlaces));
  ++iProtected;
  int *piSNZxRow = INTEGER(sexpTmp);

  // Totals of non zero cells in S per transition
  PROTECT(sexpTmp = allocVector(INTSXP, iTransitions));
  ++iProtected;
  int *piSNZxRowTot = INTEGER(sexpTmp);

  for (iTransition = 0; iTransition < iTransitions; iTransition++) {
    int iPreNZxRow_col = 0;
    int iSNZxRow_col = 0;
    for (iPlace = 0; iPlace < iPlaces; iPlace++) {
      if (piPre[iTransition + iTransitions * iPlace]) {
	piPreNZxRow[iTransition + iTransitions * iPreNZxRow_col] = iPlace;
	iPreNZxRow_col++;
      }
      if (piS[iTransition + iTransitions * iPlace]) {
	piSNZxRow[iTransition + iTransitions * iSNZxRow_col] = iPlace;
	iSNZxRow_col++;
      }
    }
    piPreNZxRowTot[iTransition] = iPreNZxRow_col;
    piSNZxRowTot[iTransition] = iSNZxRow_col;
  }

  // Hazards that need to be recalculated if a given transition has happened
  PROTECT(sexpTmp = allocMatrix(INTSXP, iTransitions + 1, iTransitions));
  ++iProtected;
  int *piHazardsToModxRow = INTEGER(sexpTmp);

  // Totals of hazards to recalculate for each transition that has happened
  PROTECT(sexpTmp = allocVector(INTSXP, iTransitions + 1));
  ++iProtected;
  int *piHazardsToModxRowTot = INTEGER(sexpTmp);

  for(iTransition = 0; iTransition < iTransitions; iTransition++) {
    int iHazardToCompTot = 0;
    for(iPlace = 0; iPlace < iPlaces; iPlace++) {
      if (piS[iTransition + iTransitions * iPlace]) {
	// Identify the transitions that need the hazards recalculated
	for(iTransitionPtr2 = 0; iTransitionPtr2 < piPreNZxColTot[iPlace]; iTransitionPtr2++) {
	  iTransition2 = piPreNZxCol[iTransitionPtr2 + iTransitions * iPlace];
	  int iAddThis = TRUE;
	  for (k = 0; k < iHazardToCompTot; k++) {
	    if(piHazardsToModxRow[iTransition + (iTransitions + 1) * k] == iTransition2) {
	      iAddThis = FALSE;
	      break;
	    }
	  }	    
	  if (iAddThis)
	    piHazardsToModxRow[iTransition + (iTransitions + 1) * iHazardToCompTot++] = iTransition2;
	}
      }
    }
    piHazardsToModxRowTot[iTransition] = iHazardToCompTot;
  }

  double dAcumHazard = 0;
  // Hazard vector
  PROTECT(sexpTmp = allocVector(REALSXP, iTransitions));
  ++iProtected;
  double *pdHazard = REAL(sexpTmp);

  // For the initial calculation of all hazards...
  for(iTransition = 0; iTransition < iTransitions; iTransition++) {
    piHazardsToModxRow[iTransitions + (iTransitions + 1) * iTransition] = iTransition;
    pdHazard[iTransition] = 0;
  }
  piHazardsToModxRowTot[iTransitions] = iTransitions;

  int iLastTransition = iTransitions;

  int iSteps = *INTEGER(steps);

  // Marking Matrix
  SEXP sexpMarking;
  PROTECT(sexpMarking = allocMatrix(INTSXP, iSteps, iPlaces));
  ++iProtected;
  int *piMarking = INTEGER(sexpMarking);
  
  // Setup initial state
  piTmp = INTEGER(M);
  for (iPlace = 0; iPlace < iPlaces; iPlace++)
    piMarking[/* 0 + */ iSteps * iPlace] = piTmp[iPlace];

  // Time Vector
  SEXP sexpTime;
  PROTECT(sexpTime = allocVector(REALSXP, iSteps));
  ++iProtected;
  double *pdTime = REAL(sexpTime);
  pdTime[0] = 0;

  int iStep;
  GetRNGstate();
  for (iStep = 1; iStep < iSteps; iStep++) {

    // Get hazards only for the transitions associated with
    // places whose quantities changed in the last step.
    for(iTransitionPtr = 0; iTransitionPtr < piHazardsToModxRowTot[iLastTransition]; iTransitionPtr++) {
      iTransition = piHazardsToModxRow[iLastTransition + (iTransitions + 1) * iTransitionPtr];

      double dNewHazard = pdH[iTransition];
      for(iPlacePtr = 0; iPlacePtr < piPreNZxRowTot[iTransition]; iPlacePtr++) {
	iPlace = piPreNZxRow[iTransition + iTransitions * iPlacePtr];
	for (k = 0; k < piPre[iTransition + iTransitions * iPlace]; k++)
	  dNewHazard *= (double)(piMarking[(iStep - 1) + iSteps * iPlace] - k) / (double)(k+1);
      }
      dAcumHazard += dNewHazard - pdHazard[iTransition];
      pdHazard[iTransition] = dNewHazard;
    }


    // Check if there are possible transitions to perform.
    // If not, end prematurely.
    if (dAcumHazard < HAZARD_VALUE_TO_IGNORE)
      break;

    // Get Time to transition
    // For rexp the parameter is scale, not rate.
    // This is why the reciprocal of the rate is sent.
    pdTime[iStep] = pdTime[iStep - 1] + rexp(1 / dAcumHazard);

    double dPartialAcumHazard = 0;

    // Find out which transition happened
    double dRnd = runif(0, dAcumHazard);
    int iTransition, iPlace;
    for(iTransition = 0; iTransition < iTransitions; iTransition++) {
      if (dRnd < (dPartialAcumHazard += pdHazard[iTransition])) {
	iLastTransition = iTransition;
	iTmp = 0;
	for(iPlace = 0; iPlace < iPlaces; iPlace++) {
	  // Update the state
	  iTmp += (piMarking[iStep + iSteps * iPlace] = 
		   piMarking[(iStep - 1) + iSteps * iPlace] + 
		   piS[iTransition + iTransitions * iPlace]);

	}
	if (iTmp > PRODUCTS_MAXIMUM_AMOUNT)
	  goto EXIT_LOOP;
	break;
      }
    }
  }
 EXIT_LOOP:
  PutRNGstate();
  
  if (iStep < iSteps) {
    int iStepsNew = iStep;

    PROTECT(sexpMarking = allocMatrix(INTSXP, iStepsNew, iPlaces));
    ++iProtected;
    int *piMarkingNew = INTEGER(sexpMarking);
    for(iPlace = 0; iPlace < iPlaces; iPlace++)
      for (iStep = 0; iStep < iStepsNew; iStep++)
	piMarkingNew[iStep + iStepsNew * iPlace] = piMarking[iStep + iSteps * iPlace];
    
    PROTECT(sexpTime = allocVector(REALSXP, iStepsNew));
    ++iProtected;
    double *pdTimeNew = REAL(sexpTime);
    for (iStep = 0; iStep < iStepsNew; iStep++)
      pdTimeNew[iStep] = pdTime[iStep];
  }
  
  SEXP sexpAns;
  PROTECT(sexpAns = allocVector(VECSXP, 2));
  ++iProtected;

  SET_VECTOR_ELT(sexpAns, 0, sexpTime);
  SET_VECTOR_ELT(sexpAns, 1, sexpMarking);

#ifdef RB_TIME
  c1 = clock();
  Rprintf ("Elapsed CPU time: %f\n", (float) (c1 - c0)/CLOCKS_PER_SEC);
#endif

  UNPROTECT(iProtected);
  return(sexpAns);
}

typedef enum {
  HZ_DOUBLE,
  HZ_CFUNCTION,
  HZ_RFUNCTION
} HZ_type;


SEXP GillespieOptimDirect(SEXP pre, SEXP post, SEXP h, SEXP M, SEXP T, SEXP delta,
			  SEXP runs, SEXP place, SEXP transition, SEXP rho)
{
  int k;

#ifdef RB_TIME
  clock_t c0, c1;
  c0 = clock();
#endif

  // Get dimensions of pre
  int *piTmp = INTEGER(getAttrib(pre, R_DimSymbol));
  int iTransitions = piTmp[0], iPlaces = piTmp[1];

  int *piPre = INTEGER(pre), *piPost = INTEGER(post);

  SEXP sexpTmp;

  int iTransition, iPlace, iTransitionPtr, iPlacePtr,
    iTransition2, iTransitionPtr2;

  // Find out which elements of h are doubles and which functions
  SEXP sexpFunction;
  PROTECT(sexpFunction = allocVector(VECSXP, iTransitions));
  double *pdH = (double *) R_alloc(iTransitions, sizeof(double));
  DL_FUNC *pCFunction = (DL_FUNC *) R_alloc(iTransitions, sizeof(DL_FUNC *));
  int *piHzType = (int *) R_alloc(iTransitions, sizeof(int));
  for (iTransition = 0; iTransition < iTransitions; iTransition++) {
    if (inherits(sexpTmp = VECTOR_ELT(h, iTransition), "NativeSymbol")) {
      pCFunction[iTransition] = (void *) R_ExternalPtrAddr(sexpTmp);
      piHzType[iTransition] = HZ_CFUNCTION;    
    } else if (isNumeric(sexpTmp)){
      pdH[iTransition] = REAL(sexpTmp)[0];
      piHzType[iTransition] = HZ_DOUBLE;
    } else  if (isFunction(sexpTmp)) {
      SET_VECTOR_ELT(sexpFunction, iTransition, lang1(sexpTmp));
      piHzType[iTransition] = HZ_RFUNCTION;
    } else {
      error("Unrecongnized transition function type\n");
    }
  }

  // Setup Matrix S
  int *piS = (int *) R_alloc(iTransitions * iPlaces, sizeof(int));

  // Position of non zero cells in pre per transition
  int *piPreNZxRow = (int *) R_alloc(iTransitions * iPlaces, sizeof(int));

  // Totals of non zero cells in pre per transition
  int *piPreNZxRowTot = (int *) R_alloc(iTransitions, sizeof(int));

  // Position of non zero cells in S per transition
  int *piSNZxRow = (int *) R_alloc(iTransitions * iPlaces, sizeof(int));

  // Totals of non zero cells in S per transition
  int *piSNZxRowTot = (int *) R_alloc(iTransitions, sizeof(int));

  int *piOrderedTransition = (int *) R_alloc(iTransitions, sizeof(int));

  for (iTransition = 0; iTransition < iTransitions; iTransition++) {
    int iPreNZxRow_col = 0;
    int iSNZxRow_col = 0;
    for (iPlace = 0; iPlace < iPlaces; iPlace++) {
      if (piPre[iTransition + iTransitions * iPlace]) {
	piPreNZxRow[iTransition + iTransitions * iPreNZxRow_col++] = iPlace;
      }
      if ((piS[iTransition + iTransitions * iPlace] = 
	   piPost[iTransition + iTransitions * iPlace] - piPre[iTransition + iTransitions * iPlace])) {
	piSNZxRow[iTransition + iTransitions * iSNZxRow_col++] = iPlace;
      }
    }
    piPreNZxRowTot[iTransition] = iPreNZxRow_col;
    piSNZxRowTot[iTransition] = iSNZxRow_col;
  }

  // Position of non zero cells in pre per place
  int *piPreNZxCol = (int *) R_alloc(iTransitions * iPlaces, sizeof(int));

  // Totals of non zero cells in pre per place
  int *piPreNZxColTot = (int *) R_alloc(iPlaces, sizeof(int));

  for (iPlace = 0; iPlace < iPlaces; iPlace++) {
    int iPreNZxCol_row = 0;
    for (iTransition = 0; iTransition < iTransitions; iTransition++) {
      if (piPre[iTransition + iTransitions * iPlace]) {
	piPreNZxCol[iPreNZxCol_row++ + iTransitions * iPlace] = iTransition;
      }
    }
    piPreNZxColTot[iPlace] = iPreNZxCol_row;
  }

  // Hazards that need to be recalculated if a given transition has happened
  int *piHazardsToModxRow = (int *) R_alloc((iTransitions + 1) * iTransitions, sizeof(int));

  // Totals of hazards to recalculate for each transition that has happened
  int *piHazardsToModxRowTot = (int *) R_alloc(iTransitions + 1, sizeof(int));
  
  for(iTransition = 0; iTransition < iTransitions; iTransition++) {
    int iHazardToCompTot = 0;
    for(iPlace = 0; iPlace < iPlaces; iPlace++) {
      if (piS[iTransition + iTransitions * iPlace]) {
	// Identify the transitions that need the hazards recalculated
	for(iTransitionPtr2 = 0; iTransitionPtr2 < piPreNZxColTot[iPlace]; iTransitionPtr2++) {
	  iTransition2 = piPreNZxCol[iTransitionPtr2 + iTransitions * iPlace];
	  int iAddThis = TRUE;
	  for (k = 0; k < iHazardToCompTot; k++) {
	    if(piHazardsToModxRow[iTransition + (iTransitions + 1) * k] == iTransition2) {
	      iAddThis = FALSE;
	      break;
	    }
	  }	    
	  if (iAddThis)
	    piHazardsToModxRow[iTransition + (iTransitions + 1) * iHazardToCompTot++] = iTransition2;
	}
      }
    }
    piHazardsToModxRowTot[iTransition] = iHazardToCompTot;
  }
  // For the initial calculation of all hazards...
  for(iTransition = 0; iTransition < iTransitions; iTransition++) {
    piHazardsToModxRow[iTransitions + (iTransitions + 1) * iTransition] = iTransition;
  }
  piHazardsToModxRowTot[iTransitions] = iTransitions;

  SEXP sexpCrntMarking;
  PROTECT(sexpCrntMarking = allocVector(REALSXP, iPlaces));
  double *pdCrntMarking = REAL(sexpCrntMarking);

  double dDelta = *REAL(delta);
  int iTotalSteps, iSectionSteps;
  double dT = 0;
  void *pCManage_time = 0;
  SEXP sexpRManage_time = 0;
  if (inherits(T, "NativeSymbol")) {
    pCManage_time = (void *) R_ExternalPtrAddr(T);
    dT = ((double(*)(double, double *)) pCManage_time)(-1, pdCrntMarking);
  } else if (isNumeric(T)){
    dT = *REAL(T);
  } else  if (isFunction(T)) {
    PROTECT(sexpRManage_time = lang1(T));

    defineVar(install("y"), sexpCrntMarking, rho);
    PROTECT(sexpTmp = allocVector(REALSXP, 1));
    *REAL(sexpTmp) = -1;
    defineVar(install("StartTime"), sexpTmp, rho);
    UNPROTECT_PTR(sexpTmp);
    dT = *REAL(VECTOR_ELT(eval(sexpRManage_time, rho),0));
  } else {
    error("Unrecognized time function type\n");
  }
  
  iTotalSteps = iSectionSteps = (int)(dT / dDelta) + 1;

  int iRun, iRuns = *INTEGER(runs);

  // Hazard vector
  double *pdHazard = (double *) R_alloc(iTransitions, sizeof(double));

  SEXP sexpRun;
  PROTECT(sexpRun = allocVector(VECSXP, iRuns));

  int iTotalUsedRandomNumbers = 0;

  // DiscTime Vector
  SEXP sexpD_time;
  PROTECT(sexpD_time = allocVector(REALSXP, iTotalSteps));
  double *pdDiscTime = REAL(sexpD_time);
  pdDiscTime[0] = 0;

  SEXP sexpMarkingRowNames;
  PROTECT(sexpMarkingRowNames = allocVector(INTSXP, iTotalSteps));
  piTmp = INTEGER(sexpMarkingRowNames);
  for (k = 0; k < iTotalSteps; k++)
    piTmp[k] = k+1;

  double **ppdMarking = (double **) R_alloc(iPlaces, sizeof(double *));

  GetRNGstate();
  for (iRun = 0; iRun < iRuns; iRun++) {

    int iUsedRandomNumbers = 0;
    Rprintf("%d ", iRun+1);

    // Totals for kind of transition vector
    SEXP sexpTotXTransition;
    PROTECT(sexpTotXTransition = allocVector(INTSXP, iTransitions));
    int *piTotTransitions = INTEGER(sexpTotXTransition);
  
    for(iTransition = 0; iTransition < iTransitions; iTransition++) {
      piTotTransitions[iTransition] = 0;

      piOrderedTransition[iTransition] = iTransition;
    }
  
    int iTillResort = 1000, iTotResort = 0;

    SEXP sexpMarking;
    PROTECT(sexpMarking = allocVector(VECSXP, iPlaces));
    //setAttrib(sexpMarking, R_NamesSymbol, place);
    //setAttrib(sexpMarking, R_RowNamesSymbol, sexpMarkingRowNames);
    //setAttrib(sexpMarking, R_ClassSymbol, ScalarString(mkChar("data.frame")));

    // Setup initial state
    double *pdTmp = REAL(M);
    for (iPlace = 0; iPlace < iPlaces; iPlace++) {
      SET_VECTOR_ELT(sexpMarking, iPlace, sexpTmp = allocVector(REALSXP, iTotalSteps));
      ppdMarking[iPlace] = REAL(sexpTmp);

      pdCrntMarking[iPlace] = pdTmp[iPlace];
    }

    double dAcumHazard = 0;
    for(iTransition = 0; iTransition < iTransitions; iTransition++) {
      pdHazard[iTransition] = 0;
    }
      
    double dTime = 0, dTarget = 0;
    int iTotTransitions = 0;

    int iStep = 0;
    double dNewHazard = 0;
    do {
      if (iStep) {
	--iStep;
	for(iPlace = 0; iPlace < iPlaces; iPlace++) {
	  pdCrntMarking[iPlace] = ppdMarking[iPlace][iStep];
	}
      }
      
      if (pCManage_time || sexpRManage_time) {
	double dEnd = 0;
	if (pCManage_time) {
	  dEnd = ((double(*)(double, double *)) pCManage_time)(dTarget, pdCrntMarking);
	} else {
	  defineVar(install("y"), sexpCrntMarking, rho);
	  PROTECT(sexpTmp = allocVector(REALSXP, 1));
	  *REAL(sexpTmp) = dTarget;
	  defineVar(install("StartTime"), sexpTmp, rho);
	  UNPROTECT_PTR(sexpTmp);

	  sexpTmp = eval(sexpRManage_time, rho);
	  dEnd = *REAL(VECTOR_ELT(sexpTmp,0));
	  for(iPlace = 0; iPlace < iPlaces; iPlace++) {
	    pdCrntMarking[iPlace] = REAL(VECTOR_ELT(sexpTmp,1))[iPlace];
	  }
	}
	iSectionSteps = (int)(dEnd / dDelta) + 1;
      }
      dTime = dTarget;
      
      // For the calculation of all hazards...
      int iLastTransition = iTransitions;
      do {    
	// Get hazards only for the transitions associated with
	// places whose quantities changed in the last step.
	for(iTransitionPtr = 0; iTransitionPtr < piHazardsToModxRowTot[iLastTransition]; iTransitionPtr++) {
	  iTransition = piHazardsToModxRow[iLastTransition + (iTransitions + 1) * iTransitionPtr];
	  switch(piHzType[iTransition]) {
	  case HZ_DOUBLE:
	    dNewHazard = pdH[iTransition];
	    for(iPlacePtr = 0; iPlacePtr < piPreNZxRowTot[iTransition]; iPlacePtr++) {
	      iPlace = piPreNZxRow[iTransition + iTransitions * iPlacePtr];
	      for (k = 0; k < piPre[iTransition + iTransitions * iPlace]; k++)
		dNewHazard *= (pdCrntMarking[iPlace] - k) / (double)(k+1);
	    }
	    break;
	  case HZ_CFUNCTION:	
	    dNewHazard = ((double(*)(double *)) pCFunction[iTransition])(pdCrntMarking);
	    break;
	  case HZ_RFUNCTION:
	    defineVar(install("y"), sexpCrntMarking, rho);
	    dNewHazard = REAL(eval(VECTOR_ELT(sexpFunction, iTransition), rho))[0];
	    break;
	  }
	  dAcumHazard += dNewHazard - pdHazard[iTransition];
	  pdHazard[iTransition] = dNewHazard;
	}
	
	// Get Time to transition
	if (dAcumHazard > 0) {
	  dTime += exp_rand() / dAcumHazard;
	  //dTime -= log(unif_rand()) / dAcumHazard;
	  iUsedRandomNumbers++;
	} else
	  dTime = dT + 1;	
	
	while (dTime >= dTarget) {
	  if (!iRun)
	    pdDiscTime[iStep] = dTarget;
	  // Update the state for the fixed incremented time.
	  for(iPlace = 0; iPlace < iPlaces; iPlace++)
	    ppdMarking[iPlace][iStep] = pdCrntMarking[iPlace];
	  if (++iStep >= iSectionSteps)
	    goto EXIT_LOOP;
	  dTarget += dDelta;
	  // Allow user interruption
	  R_CheckUserInterrupt();
	}
	
	while (!--iTillResort) {
	  quicksort(piOrderedTransition, piTotTransitions, 0, iTransitions-1);
	  switch (iTotResort++) {
	  case 0:
	    iTillResort = 10000;
	    break;
	  case 1:
	    iTillResort = 100000;
	    break;
	  default:
	    iTillResort = 1000000;
	  }
	}
	double dPartialAcumHazard = 0;
	// Find out which transition happened
	double dRnd = runif(0, dAcumHazard);
	iUsedRandomNumbers++;
	for(iTransitionPtr = 0; iTransitionPtr < iTransitions; iTransitionPtr++) {
	  iTransition = piOrderedTransition[iTransitionPtr];
	  if (dRnd < (dPartialAcumHazard += pdHazard[iTransition])) {
	    piTotTransitions[iLastTransition = iTransition]++;
	    for(iPlacePtr = 0; iPlacePtr < piSNZxRowTot[iTransition]; iPlacePtr++) {
	      iPlace = piSNZxRow[iTransition + iTransitions * iPlacePtr];
	      
	      // Update the state
	      pdCrntMarking[iPlace] += piS[iTransition + iTransitions * iPlace];
	    }
	    break;
	  }
	}
      } while (++iTotTransitions);
    EXIT_LOOP:;
      Rprintf(".");
    } while (iSectionSteps < iTotalSteps);
    iTotalUsedRandomNumbers += iUsedRandomNumbers;
    Rprintf("\t%d\t%d\t%d", iTotTransitions, iUsedRandomNumbers, iTotalUsedRandomNumbers);
#ifdef RB_SUBTIME
    c1 = clock();
    Rprintf ("\t To go: ");
    PrintfTime((double) (c1 - c0)/CLOCKS_PER_SEC/(iRun+1)*(iRuns-iRun-1));
#endif
    Rprintf ("\n");
  
    /*
    for(iTransition = 0; iTransition < iTransitions; iTransition++) {
      Rprintf("%d\t%d\n", piOrderedTransition[iTransition], piTotTransitions[piOrderedTransition[iTransition]]);
    }
    */

    SEXP sexpTotTransitions;
    PROTECT(sexpTotTransitions = allocVector(INTSXP, 1));
    INTEGER(sexpTotTransitions)[0] = iTotTransitions;

    SEXP sexpThisRun;
    PROTECT(sexpThisRun = allocVector(VECSXP, 3));

    SET_VECTOR_ELT(sexpThisRun, 0, sexpMarking);
    UNPROTECT_PTR(sexpMarking);
    SET_VECTOR_ELT(sexpThisRun, 1, sexpTotXTransition);
    UNPROTECT_PTR(sexpTotXTransition);
    SET_VECTOR_ELT(sexpThisRun, 2, sexpTotTransitions);
    UNPROTECT_PTR(sexpTotTransitions);

    SEXP sexpNames;
    PROTECT(sexpNames = allocVector(VECSXP, 3));
    SET_VECTOR_ELT(sexpNames, 0, mkChar("M"));
    SET_VECTOR_ELT(sexpNames, 1, mkChar("transitions"));
    SET_VECTOR_ELT(sexpNames, 2, mkChar("tot.transitions"));
    setAttrib(sexpThisRun, R_NamesSymbol, sexpNames);
    UNPROTECT_PTR(sexpNames);

    SET_VECTOR_ELT(sexpRun, iRun, sexpThisRun);
    UNPROTECT_PTR(sexpThisRun);
  }
  PutRNGstate();

  SEXP sexpAns;
  PROTECT(sexpAns = allocVector(VECSXP, 4));
  SET_VECTOR_ELT(sexpAns, 0, place);
  SET_VECTOR_ELT(sexpAns, 1, transition);
  SET_VECTOR_ELT(sexpAns, 2, sexpD_time);
  UNPROTECT_PTR(sexpD_time);
  SET_VECTOR_ELT(sexpAns, 3, sexpRun);
  UNPROTECT_PTR(sexpRun);

  SEXP sexpNames;
  PROTECT(sexpNames = allocVector(VECSXP, 4));
  SET_VECTOR_ELT(sexpNames, 0, mkChar("place"));
  SET_VECTOR_ELT(sexpNames, 1, mkChar("transition"));
  SET_VECTOR_ELT(sexpNames, 2, mkChar("dt"));
  SET_VECTOR_ELT(sexpNames, 3, mkChar("run"));
  setAttrib(sexpAns, R_NamesSymbol, sexpNames);
  UNPROTECT_PTR(sexpNames);

#ifdef RB_TIME
  c1 = clock();
  Rprintf ("Elapsed CPU time: %f\n", (float) (c1 - c0)/CLOCKS_PER_SEC);
#endif

  if (sexpRManage_time)
    UNPROTECT_PTR(sexpRManage_time);
  UNPROTECT_PTR(sexpFunction);
  UNPROTECT_PTR(sexpMarkingRowNames);
  UNPROTECT_PTR(sexpCrntMarking);
  UNPROTECT_PTR(sexpAns);
  return(sexpAns);
}


SEXP rb_gillespie_orig(SEXP pre, SEXP post, SEXP h, SEXP M, SEXP steps)
{
  int i, j, k, iProtected = 0;
  double dTmp;
  
#ifdef RB_TIME
  clock_t c0, c1;
  c0 = clock();
#endif
  
  // Get dimensions of pre
  int *piTmp = INTEGER(getAttrib(pre, R_DimSymbol));
  int iTransitions = piTmp[0], iPlaces = piTmp[1];

  int *piPre = INTEGER(pre), *piPost = INTEGER(post);

  double *pdH = REAL(h);

  // Setup Matrix S
  SEXP sexpTmp;
  PROTECT(sexpTmp = allocMatrix(INTSXP, iTransitions, iPlaces));
  ++iProtected;
  int *piS = INTEGER(sexpTmp);
  // Do it as vector, no matrix (same result)
  for (i = 0; i < iTransitions * iPlaces; i++) {
      piS[i] = piPost[i] - piPre[i];
  }

  int iSteps = *INTEGER(steps);

  // Marking Matrix
  SEXP sexpMarking;
  PROTECT(sexpMarking = allocMatrix(INTSXP, iSteps, iPlaces));
  ++iProtected;
  int *piMarking = INTEGER(sexpMarking);
  
  // Setup initial state
  piTmp = INTEGER(M);
  for (j = 0; j < iPlaces; j++)
    piMarking[/* 0 + */ iSteps * j] = piTmp[j];

  // Time Vector
  SEXP sexpTime;
  PROTECT(sexpTime = allocVector(REALSXP, iSteps));
  ++iProtected;
  double *pdTime = REAL(sexpTime);
  pdTime[0] = 0;

  // Accumulated Hazard vector
  PROTECT(sexpTmp = allocVector(REALSXP, iTransitions));
  ++iProtected;
  double *pdAcumHazard = REAL(sexpTmp);

  int iStep;
  GetRNGstate();
  for (iStep = 1; iStep < iSteps; iStep++) {

    // Get transition hazards    
    for(i = 0; i < iTransitions; i++) {
      dTmp = pdH[i];
      for(j = 0; j < iPlaces; j++) {
	// Cycle throw Pre to see how many molecules of each type are reacting
	// (monomers, dimers, ...)
	for (k = 0; k < piPre[i + iTransitions*j]; k++)
	  dTmp *= (double)(piMarking[(iStep - 1) + iSteps * j] - k) / (double)(k+1);
      }
      if (i > 0)
	dTmp += pdAcumHazard[i - 1];
      pdAcumHazard[i] = dTmp;
    }
    // Check if there are possible transitions to perform.
    // If not, end prematurely.
    if (pdAcumHazard[iTransitions - 1] < 1e-10)
      break;

    // Get Time to transition
    // For rexp the parameter is scale, not rate.
    // This is why the reciprocal of the rate is sent.
    pdTime[iStep] = pdTime[iStep - 1] + rexp(1 / pdAcumHazard[iTransitions - 1]);

    // Find out which transition happened
    dTmp = runif(0, pdAcumHazard[iTransitions- 1]);
    for(i = 0; i < iTransitions; i++) {
      if (dTmp < pdAcumHazard[i])
	break;
    }
    for(j = 0; j < iPlaces; j++) {
      // Update the state
      piMarking[iStep + iSteps * j] = 
	piMarking[(iStep - 1) + iSteps * j] + piS[i + iTransitions * j];
    }
  }
  PutRNGstate();
  
  if (iStep < iSteps) {
    int iStepsNew = iStep;

    PROTECT(sexpMarking = allocMatrix(INTSXP, iStepsNew, iPlaces));
    ++iProtected;
    int *piMarkingNew = INTEGER(sexpMarking);
    for(j = 0; j < iPlaces; j++)
      for (iStep = 0; iStep < iStepsNew; iStep++)
	piMarkingNew[iStep + iStepsNew * j] = piMarking[iStep + iSteps * j];
    
    PROTECT(sexpTime = allocVector(REALSXP, iStepsNew));
    ++iProtected;
    double *pdTimeNew = REAL(sexpTime);
    for (iStep = 0; iStep < iStepsNew; iStep++)
      pdTimeNew[iStep] = pdTime[iStep];
  }
  
  SEXP sexpAns;
  PROTECT(sexpAns = allocVector(VECSXP, 2));
  ++iProtected;

  SET_VECTOR_ELT(sexpAns, 0, sexpTime);
  SET_VECTOR_ELT(sexpAns, 1, sexpMarking);

#ifdef RB_TIME
  c1 = clock();
  Rprintf ("Elapsed CPU time: %f\n", (float) (c1 - c0)/CLOCKS_PER_SEC);
#endif

  UNPROTECT(iProtected);
  return(sexpAns);
}


SEXP rb_gillespie_d_orig(SEXP pre, SEXP post, SEXP h, SEXP M, SEXP T, SEXP delta)
{
  int i, j, k, iTmp, iProtected = 0;
  double dTmp;

#ifdef RB_TIME
  clock_t c0, c1;
  c0 = clock();
#endif

  // Get dimensions of pre
  int *piTmp = INTEGER(getAttrib(pre, R_DimSymbol));
  int iTransitions = piTmp[0], iPlaces = piTmp[1];

  int *piPre = INTEGER(pre), *piPost = INTEGER(post);

  double *pdH = REAL(h);

  // Setup Matrix S
  SEXP sexpTmp;
  PROTECT(sexpTmp = allocMatrix(INTSXP, iTransitions, iPlaces));
  ++iProtected;
  int *piS = INTEGER(sexpTmp);
  // Do it as vector, no matrix (same result)
  for (i = 0; i < iTransitions * iPlaces; i++) {
      piS[i] = piPost[i] - piPre[i];
  }

  double dT = *REAL(T), dDelta = *REAL(delta);
  int iSteps = (int)(dT / dDelta) + 1;

  // Marking Matrix
  SEXP sexpMarking;
  PROTECT(sexpMarking = allocMatrix(INTSXP, iSteps, iPlaces));
  ++iProtected;
  int *piMarking = INTEGER(sexpMarking);

  PROTECT(sexpTmp = allocVector(INTSXP, iPlaces));
  ++iProtected;
  int *piCrntMarking = INTEGER(sexpTmp);
  // Setup initial state
  piTmp = INTEGER(M);
  for (j = 0; j < iPlaces; j++)
    piCrntMarking[j] = piTmp[j];

  // DiscTime Vector
  SEXP sexpDiscTime;
  PROTECT(sexpDiscTime = allocVector(REALSXP, iSteps));
  ++iProtected;
  double *pdDiscTime = REAL(sexpDiscTime);
  pdDiscTime[0] = 0;

  double dTime = 0, dTarget = 0;
  int iTotTransitions = 0;

  // Accumulated Hazard vector
  PROTECT(sexpTmp = allocVector(REALSXP, iTransitions));
  ++iProtected;
  double *pdAcumHazard = REAL(sexpTmp);

  int iStep = 0;
  GetRNGstate();
  while (++iTotTransitions) {
    // Get transition hazards    
    for(i = 0; i < iTransitions; i++) {
      dTmp = pdH[i];
      for(j = 0; j < iPlaces; j++) {
	// Cycle throw Pre to see how many molecules of each type are reacting
	// (monomers, dimers, ...)
	for (k = 0; k < piPre[i + iTransitions*j]; k++)
	  dTmp *= (double)(piCrntMarking[j] - k) / (double)(k+1);
      }
      if (i > 0)
	dTmp += pdAcumHazard[i - 1];
      pdAcumHazard[i] = dTmp;
    }
    // Check if there are possible transitions to perform.
    // If not, set maximum time + 1.
    if (pdAcumHazard[iTransitions - 1] < 1e-10)
      dTime = dT + 1;
    else {
      // Get Time to transition
      // For rexp the parameter is scale, not rate.
      // This is why the reciprocal of the rate is sent.
      dTime += rexp(1 / pdAcumHazard[iTransitions - 1]);
    }

    while (dTime >= dTarget) {
      pdDiscTime[iStep] = dTarget;
      // Update the state for the fixed incremented time.
      for(j = 0; j < iPlaces; j++) {
	piMarking[iStep + iSteps * j] = piCrntMarking[j];
      }
      if (++iStep >= iSteps) {
	goto EXIT_LOOP;
      }
      dTarget += dDelta;
    }
    
    // Find out which transition happened
    dTmp = runif(0, pdAcumHazard[iTransitions - 1]);
    for(i = 0; i < iTransitions; i++) {
      if (dTmp < pdAcumHazard[i])
	break;
    }
    // Update the state and check if maximum number of places
    // has been reached
    iTmp = 0;
    for(j = 0; j < iPlaces; j++) {
      iTmp += (piCrntMarking[j] += piS[i + iTransitions * j]);
    }
    if (iTmp > PRODUCTS_MAXIMUM_AMOUNT)
      goto EXIT_LOOP;
  }
 EXIT_LOOP:
  PutRNGstate();
  
  SEXP sexpTotTransitions;
  PROTECT(sexpTotTransitions = allocVector(INTSXP, 1));
  ++iProtected;
  INTEGER(sexpTotTransitions)[0] = iTotTransitions;

  SEXP sexpAns;
  PROTECT(sexpAns = allocVector(VECSXP, 3));
  ++iProtected;

  SET_VECTOR_ELT(sexpAns, 0, sexpDiscTime);
  SET_VECTOR_ELT(sexpAns, 1, sexpMarking);
  SET_VECTOR_ELT(sexpAns, 2, sexpTotTransitions);

#ifdef RB_TIME
  c1 = clock();
  Rprintf ("Elapsed CPU time: %f\n", (float) (c1 - c0)/CLOCKS_PER_SEC);
#endif

  UNPROTECT(iProtected);
  return(sexpAns);
}
